"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.radarrScanner = void 0;
const radarr_1 = __importDefault(require("../../../api/servarr/radarr"));
const baseScanner_1 = __importDefault(require("../../../lib/scanners/baseScanner"));
const settings_1 = require("../../../lib/settings");
const lodash_1 = require("lodash");
class RadarrScanner extends baseScanner_1.default {
    constructor() {
        super('Radarr Scan', { bundleSize: 50 });
    }
    status() {
        return {
            running: this.running,
            progress: this.progress,
            total: this.items.length,
            currentServer: this.currentServer,
            servers: this.servers,
        };
    }
    async run() {
        const settings = (0, settings_1.getSettings)();
        const sessionId = this.startRun();
        try {
            this.servers = (0, lodash_1.uniqWith)(settings.radarr, (radarrA, radarrB) => {
                return (radarrA.hostname === radarrB.hostname &&
                    radarrA.port === radarrB.port &&
                    radarrA.baseUrl === radarrB.baseUrl);
            });
            for (const server of this.servers) {
                this.currentServer = server;
                if (server.syncEnabled) {
                    this.log(`Beginning to process Radarr server: ${server.name}`, 'info');
                    this.radarrApi = new radarr_1.default({
                        apiKey: server.apiKey,
                        url: radarr_1.default.buildUrl(server, '/api/v3'),
                    });
                    this.items = await this.radarrApi.getMovies();
                    await this.loop(this.processRadarrMovie.bind(this), { sessionId });
                }
                else {
                    this.log(`Sync not enabled. Skipping Radarr server: ${server.name}`);
                }
            }
            this.log('Radarr scan complete', 'info');
        }
        catch (e) {
            this.log('Scan interrupted', 'error', { errorMessage: e.message });
        }
        finally {
            this.endRun(sessionId);
        }
    }
    async processRadarrMovie(radarrMovie) {
        if (!radarrMovie.monitored && !radarrMovie.hasFile) {
            this.log('Title is unmonitored and has not been downloaded. Skipping item.', 'debug', {
                title: radarrMovie.title,
            });
            return;
        }
        try {
            const server4k = this.enable4kMovie && this.currentServer.is4k;
            await this.processMovie(radarrMovie.tmdbId, {
                is4k: server4k,
                serviceId: this.currentServer.id,
                externalServiceId: radarrMovie.id,
                externalServiceSlug: radarrMovie.titleSlug,
                title: radarrMovie.title,
                processing: !radarrMovie.hasFile,
            });
        }
        catch (e) {
            this.log('Failed to process Radarr media', 'error', {
                errorMessage: e.message,
                title: radarrMovie.title,
            });
        }
    }
}
exports.radarrScanner = new RadarrScanner();
