"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const jellyfin_1 = __importDefault(require("../api/jellyfin"));
const plexapi_1 = __importDefault(require("../api/plexapi"));
const radarr_1 = __importDefault(require("../api/servarr/radarr"));
const sonarr_1 = __importDefault(require("../api/servarr/sonarr"));
const media_1 = require("../constants/media");
const server_1 = require("../constants/server");
const datasource_1 = require("../datasource");
const Media_1 = __importDefault(require("../entity/Media"));
const MediaRequest_1 = __importDefault(require("../entity/MediaRequest"));
const User_1 = require("../entity/User");
const settings_1 = require("../lib/settings");
const logger_1 = __importDefault(require("../logger"));
const getHostname_1 = require("../utils/getHostname");
class AvailabilitySync {
    constructor() {
        this.running = false;
    }
    async run() {
        const settings = (0, settings_1.getSettings)();
        const mediaServerType = (0, settings_1.getSettings)().main.mediaServerType;
        this.running = true;
        this.plexSeasonsCache = {};
        this.jellyfinSeasonsCache = {};
        this.sonarrSeasonsCache = {};
        this.radarrServers = settings.radarr.filter((server) => server.syncEnabled);
        this.sonarrServers = settings.sonarr.filter((server) => server.syncEnabled);
        try {
            logger_1.default.info(`Starting availability sync...`, {
                label: 'Availability Sync',
            });
            const pageSize = 50;
            const userRepository = (0, datasource_1.getRepository)(User_1.User);
            // If it is plex admin is selected using plexToken if jellyfin admin is selected using jellyfinUserID
            let admin = null;
            if (mediaServerType === server_1.MediaServerType.PLEX) {
                admin = await userRepository.findOne({
                    select: { id: true, plexToken: true },
                    where: { id: 1 },
                });
            }
            else if (mediaServerType === server_1.MediaServerType.JELLYFIN ||
                mediaServerType === server_1.MediaServerType.EMBY) {
                admin = await userRepository.findOne({
                    where: { id: 1 },
                    select: ['id', 'jellyfinUserId', 'jellyfinDeviceId'],
                    order: { id: 'ASC' },
                });
            }
            switch (mediaServerType) {
                case server_1.MediaServerType.PLEX:
                    if (admin && admin.plexToken) {
                        this.plexClient = new plexapi_1.default({ plexToken: admin.plexToken });
                    }
                    else {
                        logger_1.default.error('Plex admin is not configured.');
                    }
                    break;
                case server_1.MediaServerType.JELLYFIN:
                case server_1.MediaServerType.EMBY:
                    if (admin) {
                        this.jellyfinClient = new jellyfin_1.default((0, getHostname_1.getHostname)(), settings.jellyfin.apiKey, admin.jellyfinDeviceId);
                        this.jellyfinClient.setUserId(admin.jellyfinUserId ?? '');
                        try {
                            await this.jellyfinClient.getSystemInfo();
                        }
                        catch (e) {
                            logger_1.default.error('Sync interrupted.', {
                                label: 'AvailabilitySync',
                                status: e.statusCode,
                                error: e.name,
                                errorMessage: e.errorCode,
                            });
                            this.running = false;
                            return;
                        }
                    }
                    else {
                        logger_1.default.error('Jellyfin admin is not configured.');
                        this.running = false;
                        return;
                    }
                    break;
                default:
                    logger_1.default.error('An admin is not configured.');
                    this.running = false;
                    return;
            }
            for await (const media of this.loadAvailableMediaPaginated(pageSize)) {
                if (!this.running) {
                    throw new Error('Job aborted');
                }
                // Check plex, radarr, and sonarr for that specific media and
                // if unavailable, then we change the status accordingly.
                // If a non-4k or 4k version exists in at least one of the instances, we will only update that specific version
                if (media.mediaType === 'movie') {
                    let movieExists = false;
                    let movieExists4k = false;
                    // if (mediaServerType === MediaServerType.PLEX) {
                    //   await this.mediaExistsInPlex(media, false);
                    // } else if (
                    //   mediaServerType === MediaServerType.JELLYFIN ||
                    //   mediaServerType === MediaServerType.EMBY
                    // ) {
                    //   await this.mediaExistsInJellyfin(media, false);
                    // }
                    const existsInRadarr = await this.mediaExistsInRadarr(media, false);
                    const existsInRadarr4k = await this.mediaExistsInRadarr(media, true);
                    // plex
                    if (mediaServerType === server_1.MediaServerType.PLEX) {
                        const { existsInPlex } = await this.mediaExistsInPlex(media, false);
                        const { existsInPlex: existsInPlex4k } = await this.mediaExistsInPlex(media, true);
                        if (existsInPlex || existsInRadarr) {
                            movieExists = true;
                            logger_1.default.info(`The non-4K movie [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                                label: 'AvailabilitySync',
                            });
                        }
                        if (existsInPlex4k || existsInRadarr4k) {
                            movieExists4k = true;
                            logger_1.default.info(`The 4K movie [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                                label: 'AvailabilitySync',
                            });
                        }
                    }
                    //jellyfin
                    if (mediaServerType === server_1.MediaServerType.JELLYFIN ||
                        mediaServerType === server_1.MediaServerType.EMBY) {
                        const { existsInJellyfin } = await this.mediaExistsInJellyfin(media, false);
                        const { existsInJellyfin: existsInJellyfin4k } = await this.mediaExistsInJellyfin(media, true);
                        if (existsInJellyfin || existsInRadarr) {
                            movieExists = true;
                            logger_1.default.info(`The non-4K movie [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                                label: 'AvailabilitySync',
                            });
                        }
                        if (existsInJellyfin4k || existsInRadarr4k) {
                            movieExists4k = true;
                            logger_1.default.info(`The 4K movie [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                                label: 'AvailabilitySync',
                            });
                        }
                    }
                    if (!movieExists && media.status === media_1.MediaStatus.AVAILABLE) {
                        await this.mediaUpdater(media, false, mediaServerType);
                    }
                    if (!movieExists4k && media.status4k === media_1.MediaStatus.AVAILABLE) {
                        await this.mediaUpdater(media, true, mediaServerType);
                    }
                }
                // If both versions still exist in plex, we still need
                // to check through sonarr to verify season availability
                if (media.mediaType === 'tv') {
                    let showExists = false;
                    let showExists4k = false;
                    //plex
                    const { existsInPlex, seasonsMap: plexSeasonsMap = new Map() } = await this.mediaExistsInPlex(media, false);
                    const { existsInPlex: existsInPlex4k, seasonsMap: plexSeasonsMap4k = new Map(), } = await this.mediaExistsInPlex(media, true);
                    //jellyfin
                    const { existsInJellyfin, seasonsMap: jellyfinSeasonsMap = new Map(), } = await this.mediaExistsInJellyfin(media, false);
                    const { existsInJellyfin: existsInJellyfin4k, seasonsMap: jellyfinSeasonsMap4k = new Map(), } = await this.mediaExistsInJellyfin(media, true);
                    const { existsInSonarr, seasonsMap: sonarrSeasonsMap } = await this.mediaExistsInSonarr(media, false);
                    const { existsInSonarr: existsInSonarr4k, seasonsMap: sonarrSeasonsMap4k, } = await this.mediaExistsInSonarr(media, true);
                    //plex
                    if (mediaServerType === server_1.MediaServerType.PLEX) {
                        if (existsInPlex || existsInSonarr) {
                            showExists = true;
                            logger_1.default.info(`The non-4K show [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                                label: 'AvailabilitySync',
                            });
                        }
                    }
                    if (mediaServerType === server_1.MediaServerType.PLEX) {
                        if (existsInPlex4k || existsInSonarr4k) {
                            showExists4k = true;
                            logger_1.default.info(`The 4K show [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                                label: 'AvailabilitySync',
                            });
                        }
                    }
                    //jellyfin
                    if (mediaServerType === server_1.MediaServerType.JELLYFIN ||
                        mediaServerType === server_1.MediaServerType.EMBY) {
                        if (existsInJellyfin || existsInSonarr) {
                            showExists = true;
                            logger_1.default.info(`The non-4K show [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                                label: 'AvailabilitySync',
                            });
                        }
                    }
                    if (mediaServerType === server_1.MediaServerType.JELLYFIN ||
                        mediaServerType === server_1.MediaServerType.EMBY) {
                        if (existsInJellyfin4k || existsInSonarr4k) {
                            showExists4k = true;
                            logger_1.default.info(`The 4K show [TMDB ID ${media.tmdbId}] still exists. Preventing removal.`, {
                                label: 'AvailabilitySync',
                            });
                        }
                    }
                    // Here we will create a final map that will cross compare
                    // with plex and sonarr. Filtered seasons will go through
                    // each season and assume the season does not exist. If Plex or
                    // Sonarr finds that season, we will change the final seasons value
                    // to true.
                    const filteredSeasonsMap = new Map();
                    media.seasons
                        .filter((season) => season.status === media_1.MediaStatus.AVAILABLE ||
                        season.status === media_1.MediaStatus.PARTIALLY_AVAILABLE)
                        .forEach((season) => filteredSeasonsMap.set(season.seasonNumber, false));
                    // non-4k
                    const finalSeasons = new Map();
                    if (mediaServerType === server_1.MediaServerType.PLEX) {
                        plexSeasonsMap.forEach((value, key) => {
                            finalSeasons.set(key, value);
                        });
                        filteredSeasonsMap.forEach((value, key) => {
                            if (!finalSeasons.has(key)) {
                                finalSeasons.set(key, value);
                            }
                        });
                        sonarrSeasonsMap.forEach((value, key) => {
                            if (!finalSeasons.has(key)) {
                                finalSeasons.set(key, value);
                            }
                        });
                    }
                    else if (mediaServerType === server_1.MediaServerType.JELLYFIN ||
                        mediaServerType === server_1.MediaServerType.EMBY) {
                        jellyfinSeasonsMap.forEach((value, key) => {
                            finalSeasons.set(key, value);
                        });
                        filteredSeasonsMap.forEach((value, key) => {
                            if (!finalSeasons.has(key)) {
                                finalSeasons.set(key, value);
                            }
                        });
                        sonarrSeasonsMap.forEach((value, key) => {
                            if (!finalSeasons.has(key)) {
                                finalSeasons.set(key, value);
                            }
                        });
                    }
                    const filteredSeasonsMap4k = new Map();
                    media.seasons
                        .filter((season) => season.status4k === media_1.MediaStatus.AVAILABLE ||
                        season.status4k === media_1.MediaStatus.PARTIALLY_AVAILABLE)
                        .forEach((season) => filteredSeasonsMap4k.set(season.seasonNumber, false));
                    // 4k
                    const finalSeasons4k = new Map();
                    if (mediaServerType === server_1.MediaServerType.PLEX) {
                        plexSeasonsMap4k.forEach((value, key) => {
                            finalSeasons4k.set(key, value);
                        });
                        filteredSeasonsMap4k.forEach((value, key) => {
                            if (!finalSeasons4k.has(key)) {
                                finalSeasons4k.set(key, value);
                            }
                        });
                        sonarrSeasonsMap4k.forEach((value, key) => {
                            if (!finalSeasons4k.has(key)) {
                                finalSeasons4k.set(key, value);
                            }
                        });
                    }
                    else if (mediaServerType === server_1.MediaServerType.JELLYFIN ||
                        mediaServerType === server_1.MediaServerType.EMBY) {
                        jellyfinSeasonsMap4k.forEach((value, key) => {
                            finalSeasons4k.set(key, value);
                        });
                        filteredSeasonsMap4k.forEach((value, key) => {
                            if (!finalSeasons4k.has(key)) {
                                finalSeasons4k.set(key, value);
                            }
                        });
                        sonarrSeasonsMap4k.forEach((value, key) => {
                            if (!finalSeasons4k.has(key)) {
                                finalSeasons4k.set(key, value);
                            }
                        });
                    }
                    if (!showExists &&
                        (media.status === media_1.MediaStatus.AVAILABLE ||
                            media.status === media_1.MediaStatus.PARTIALLY_AVAILABLE ||
                            media.seasons.some((season) => season.status === media_1.MediaStatus.AVAILABLE) ||
                            media.seasons.some((season) => season.status === media_1.MediaStatus.PARTIALLY_AVAILABLE))) {
                        await this.mediaUpdater(media, false, mediaServerType);
                    }
                    if (!showExists4k &&
                        (media.status4k === media_1.MediaStatus.AVAILABLE ||
                            media.status4k === media_1.MediaStatus.PARTIALLY_AVAILABLE ||
                            media.seasons.some((season) => season.status4k === media_1.MediaStatus.AVAILABLE) ||
                            media.seasons.some((season) => season.status4k === media_1.MediaStatus.PARTIALLY_AVAILABLE))) {
                        await this.mediaUpdater(media, true, mediaServerType);
                    }
                    // TODO: Figure out how to run seasonUpdater for each season
                    if ([...finalSeasons.values()].includes(false)) {
                        await this.seasonUpdater(media, finalSeasons, false, mediaServerType);
                    }
                    if ([...finalSeasons4k.values()].includes(false)) {
                        await this.seasonUpdater(media, finalSeasons4k, true, mediaServerType);
                    }
                }
            }
        }
        catch (ex) {
            logger_1.default.error('Failed to complete availability sync.', {
                errorMessage: ex.message,
                label: 'Availability Sync',
            });
        }
        finally {
            logger_1.default.info(`Availability sync complete.`, {
                label: 'Availability Sync',
            });
            this.running = false;
        }
    }
    cancel() {
        this.running = false;
    }
    async *loadAvailableMediaPaginated(pageSize) {
        let offset = 0;
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        const whereOptions = [
            { status: media_1.MediaStatus.AVAILABLE },
            { status: media_1.MediaStatus.PARTIALLY_AVAILABLE },
            { status4k: media_1.MediaStatus.AVAILABLE },
            { status4k: media_1.MediaStatus.PARTIALLY_AVAILABLE },
            { seasons: { status: media_1.MediaStatus.AVAILABLE } },
            { seasons: { status: media_1.MediaStatus.PARTIALLY_AVAILABLE } },
            { seasons: { status4k: media_1.MediaStatus.AVAILABLE } },
            { seasons: { status4k: media_1.MediaStatus.PARTIALLY_AVAILABLE } },
        ];
        let mediaPage;
        do {
            yield* (mediaPage = await mediaRepository.find({
                where: whereOptions,
                skip: offset,
                take: pageSize,
            }));
            offset += pageSize;
        } while (mediaPage.length > 0);
    }
    async mediaUpdater(media, is4k, mediaServerType) {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        try {
            // If media type is tv, check if a season is processing
            // to see if we need to keep the external metadata
            let isMediaProcessing = false;
            if (media.mediaType === 'tv') {
                const requestRepository = (0, datasource_1.getRepository)(MediaRequest_1.default);
                const request = await requestRepository
                    .createQueryBuilder('request')
                    .leftJoinAndSelect('request.media', 'media')
                    .where('(media.id = :id)', {
                    id: media.id,
                })
                    .andWhere('(request.is4k = :is4k AND request.status = :requestStatus)', {
                    requestStatus: media_1.MediaRequestStatus.APPROVED,
                    is4k: is4k,
                })
                    .getOne();
                if (request) {
                    isMediaProcessing = true;
                }
            }
            // Set the non-4K or 4K media to deleted
            // and change related columns to null if media
            // is not processing
            media[is4k ? 'status4k' : 'status'] = media_1.MediaStatus.DELETED;
            media[is4k ? 'serviceId4k' : 'serviceId'] = isMediaProcessing
                ? media[is4k ? 'serviceId4k' : 'serviceId']
                : null;
            media[is4k ? 'externalServiceId4k' : 'externalServiceId'] =
                isMediaProcessing
                    ? media[is4k ? 'externalServiceId4k' : 'externalServiceId']
                    : null;
            media[is4k ? 'externalServiceSlug4k' : 'externalServiceSlug'] =
                isMediaProcessing
                    ? media[is4k ? 'externalServiceSlug4k' : 'externalServiceSlug']
                    : null;
            if (mediaServerType === server_1.MediaServerType.PLEX) {
                media[is4k ? 'ratingKey4k' : 'ratingKey'] = isMediaProcessing
                    ? media[is4k ? 'ratingKey4k' : 'ratingKey']
                    : null;
            }
            else if (mediaServerType === server_1.MediaServerType.JELLYFIN ||
                mediaServerType === server_1.MediaServerType.EMBY) {
                media[is4k ? 'jellyfinMediaId4k' : 'jellyfinMediaId'] =
                    isMediaProcessing
                        ? media[is4k ? 'jellyfinMediaId4k' : 'jellyfinMediaId']
                        : null;
            }
            logger_1.default.info(`The ${is4k ? '4K' : 'non-4K'} ${media.mediaType === 'movie' ? 'movie' : 'show'} [TMDB ID ${media.tmdbId}] was not found in any ${media.mediaType === 'movie' ? 'Radarr' : 'Sonarr'} and ${mediaServerType === server_1.MediaServerType.PLEX
                ? 'plex'
                : mediaServerType === server_1.MediaServerType.JELLYFIN
                    ? 'jellyfin'
                    : 'emby'} instance. Status will be changed to deleted.`, { label: 'AvailabilitySync' });
            await mediaRepository.save(media);
        }
        catch (ex) {
            logger_1.default.debug(`Failure updating the ${is4k ? '4K' : 'non-4K'} ${media.mediaType === 'tv' ? 'show' : 'movie'} [TMDB ID ${media.tmdbId}].`, {
                errorMessage: ex.message,
                label: 'Availability Sync',
            });
        }
    }
    async seasonUpdater(media, seasons, is4k, mediaServerType) {
        const mediaRepository = (0, datasource_1.getRepository)(Media_1.default);
        // Filter out only the values that are false
        // (media that should be deleted)
        const seasonsPendingRemoval = new Map(
        // Disabled linter as only the value is needed from the filter
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        [...seasons].filter(([_, exists]) => !exists));
        // Retrieve the season keys to pass into our log
        const seasonKeys = [...seasonsPendingRemoval.keys()];
        // let isSeasonRemoved = false;
        try {
            for (const mediaSeason of media.seasons) {
                if (seasonsPendingRemoval.has(mediaSeason.seasonNumber)) {
                    mediaSeason[is4k ? 'status4k' : 'status'] = media_1.MediaStatus.DELETED;
                }
            }
            if (media.status === media_1.MediaStatus.AVAILABLE && !is4k) {
                media.status = media_1.MediaStatus.PARTIALLY_AVAILABLE;
                logger_1.default.info(`Marking the non-4K show [TMDB ID ${media.tmdbId}] as PARTIALLY_AVAILABLE because season removal has occurred.`, { label: 'Availability Sync' });
            }
            if (media.status4k === media_1.MediaStatus.AVAILABLE && is4k) {
                media.status4k = media_1.MediaStatus.PARTIALLY_AVAILABLE;
                logger_1.default.info(`Marking the 4K show [TMDB ID ${media.tmdbId}] as PARTIALLY_AVAILABLE because season removal has occurred.`, { label: 'Availability Sync' });
            }
            media.lastSeasonChange = new Date();
            await mediaRepository.save(media);
            logger_1.default.info(`The ${is4k ? '4K' : 'non-4K'} season(s) [${seasonKeys}] [TMDB ID ${media.tmdbId}] was not found in any ${media.mediaType === 'tv' ? 'Sonarr' : 'Radarr'} and ${mediaServerType === server_1.MediaServerType.PLEX
                ? 'plex'
                : mediaServerType === server_1.MediaServerType.JELLYFIN
                    ? 'jellyfin'
                    : 'emby'} instance. Status will be changed to deleted.`, { label: 'AvailabilitySync' });
        }
        catch (ex) {
            logger_1.default.debug(`Failure updating the ${is4k ? '4K' : 'non-4K'} season(s) [${seasonKeys}], TMDB ID ${media.tmdbId}.`, {
                errorMessage: ex.message,
                label: 'Availability Sync',
            });
        }
    }
    async mediaExistsInRadarr(media, is4k) {
        let existsInRadarr = false;
        // Check for availability in all of the available radarr servers
        // If any find the media, we will assume the media exists
        for (const server of this.radarrServers.filter((server) => server.is4k === is4k)) {
            const radarrAPI = new radarr_1.default({
                apiKey: server.apiKey,
                url: radarr_1.default.buildUrl(server, '/api/v3'),
            });
            try {
                let radarr;
                if (media.externalServiceId && !is4k) {
                    radarr = await radarrAPI.getMovie({
                        id: media.externalServiceId,
                    });
                }
                if (media.externalServiceId4k && is4k) {
                    radarr = await radarrAPI.getMovie({
                        id: media.externalServiceId4k,
                    });
                }
                if (radarr && radarr.hasFile) {
                    const resolution = radarr?.movieFile?.mediaInfo?.resolution?.split('x');
                    const is4kMovie = resolution?.length === 2 && Number(resolution[0]) >= 2000;
                    existsInRadarr = is4k ? is4kMovie : !is4kMovie;
                }
            }
            catch (ex) {
                if (!ex.message.includes('404')) {
                    existsInRadarr = true;
                    logger_1.default.debug(`Failure retrieving the ${is4k ? '4K' : 'non-4K'} movie [TMDB ID ${media.tmdbId}] from Radarr.`, {
                        errorMessage: ex.message,
                        label: 'Availability Sync',
                    });
                }
            }
        }
        return existsInRadarr;
    }
    async mediaExistsInSonarr(media, is4k) {
        let existsInSonarr = false;
        let preventSeasonSearch = false;
        // Check for availability in all of the available sonarr servers
        // If any find the media, we will assume the media exists
        for (const server of this.sonarrServers.filter((server) => {
            return server.is4k === is4k;
        })) {
            const sonarrAPI = new sonarr_1.default({
                apiKey: server.apiKey,
                url: sonarr_1.default.buildUrl(server, '/api/v3'),
            });
            try {
                let sonarr;
                if (media.externalServiceId && !is4k) {
                    sonarr = await sonarrAPI.getSeriesById(media.externalServiceId);
                    this.sonarrSeasonsCache[`${server.id}-${media.externalServiceId}`] =
                        sonarr.seasons;
                }
                if (media.externalServiceId4k && is4k) {
                    sonarr = await sonarrAPI.getSeriesById(media.externalServiceId4k);
                    this.sonarrSeasonsCache[`${server.id}-${media.externalServiceId4k}`] =
                        sonarr.seasons;
                }
                if (sonarr && sonarr.statistics.episodeFileCount > 0) {
                    existsInSonarr = true;
                }
            }
            catch (ex) {
                if (!ex.message.includes('404')) {
                    existsInSonarr = true;
                    preventSeasonSearch = true;
                    logger_1.default.debug(`Failure retrieving the ${is4k ? '4K' : 'non-4K'} show [TMDB ID ${media.tmdbId}] from Sonarr.`, {
                        errorMessage: ex.message,
                        label: 'Availability Sync',
                    });
                }
            }
        }
        // Here we check each season for availability
        // If the API returns an error other than a 404,
        // we will have to prevent the season check from happening
        const seasonsMap = new Map();
        if (!preventSeasonSearch) {
            const filteredSeasons = media.seasons.filter((season) => season[is4k ? 'status4k' : 'status'] === media_1.MediaStatus.AVAILABLE ||
                season[is4k ? 'status4k' : 'status'] ===
                    media_1.MediaStatus.PARTIALLY_AVAILABLE);
            for (const season of filteredSeasons) {
                const seasonExists = await this.seasonExistsInSonarr(media, season, is4k);
                if (seasonExists) {
                    seasonsMap.set(season.seasonNumber, true);
                }
            }
        }
        return { existsInSonarr, seasonsMap };
    }
    async seasonExistsInSonarr(media, season, is4k) {
        let seasonExists = false;
        // Check each sonarr instance to see if the media still exists
        // If found, we will assume the media exists and prevent removal
        // We can use the cache we built when we fetched the series with mediaExistsInSonarr
        for (const server of this.sonarrServers.filter((server) => server.is4k === is4k)) {
            let sonarrSeasons;
            if (media.externalServiceId && !is4k) {
                sonarrSeasons =
                    this.sonarrSeasonsCache[`${server.id}-${media.externalServiceId}`];
            }
            if (media.externalServiceId4k && is4k) {
                sonarrSeasons =
                    this.sonarrSeasonsCache[`${server.id}-${media.externalServiceId4k}`];
            }
            const seasonIsAvailable = sonarrSeasons?.find(({ seasonNumber, statistics }) => season.seasonNumber === seasonNumber &&
                statistics?.episodeFileCount &&
                statistics?.episodeFileCount > 0);
            if (seasonIsAvailable && sonarrSeasons) {
                seasonExists = true;
            }
        }
        return seasonExists;
    }
    // Plex
    async mediaExistsInPlex(media, is4k) {
        const ratingKey = media.ratingKey;
        const ratingKey4k = media.ratingKey4k;
        let existsInPlex = false;
        let preventSeasonSearch = false;
        // Check each plex instance to see if the media still exists
        // If found, we will assume the media exists and prevent removal
        // We can use the cache we built when we fetched the series with mediaExistsInPlex
        try {
            let plexMedia;
            if (ratingKey && !is4k) {
                plexMedia = await this.plexClient?.getMetadata(ratingKey);
                if (media.mediaType === 'tv') {
                    this.plexSeasonsCache[ratingKey] =
                        await this.plexClient?.getChildrenMetadata(ratingKey);
                }
            }
            if (ratingKey4k && is4k) {
                plexMedia = await this.plexClient?.getMetadata(ratingKey4k);
                if (media.mediaType === 'tv') {
                    this.plexSeasonsCache[ratingKey4k] =
                        await this.plexClient?.getChildrenMetadata(ratingKey4k);
                }
            }
            if (plexMedia) {
                existsInPlex = true;
            }
        }
        catch (ex) {
            if (!ex.message.includes('404')) {
                existsInPlex = true;
                preventSeasonSearch = true;
                logger_1.default.debug(`Failure retrieving the ${is4k ? '4K' : 'non-4K'} ${media.mediaType === 'tv' ? 'show' : 'movie'} [TMDB ID ${media.tmdbId}] from Plex.`, {
                    errorMessage: ex.message,
                    label: 'Availability Sync',
                });
            }
        }
        // Here we check each season in plex for availability
        // If the API returns an error other than a 404,
        // we will have to prevent the season check from happening
        if (media.mediaType === 'tv') {
            const seasonsMap = new Map();
            if (!preventSeasonSearch) {
                const filteredSeasons = media.seasons.filter((season) => season[is4k ? 'status4k' : 'status'] === media_1.MediaStatus.AVAILABLE ||
                    season[is4k ? 'status4k' : 'status'] ===
                        media_1.MediaStatus.PARTIALLY_AVAILABLE);
                for (const season of filteredSeasons) {
                    const seasonExists = await this.seasonExistsInPlex(media, season, is4k);
                    if (seasonExists) {
                        seasonsMap.set(season.seasonNumber, true);
                    }
                }
            }
            return { existsInPlex, seasonsMap };
        }
        return { existsInPlex };
    }
    async seasonExistsInPlex(media, season, is4k) {
        const ratingKey = media.ratingKey;
        const ratingKey4k = media.ratingKey4k;
        let seasonExistsInPlex = false;
        // Check each plex instance to see if the season exists
        let plexSeasons;
        if (ratingKey && !is4k) {
            plexSeasons = this.plexSeasonsCache[ratingKey];
        }
        if (ratingKey4k && is4k) {
            plexSeasons = this.plexSeasonsCache[ratingKey4k];
        }
        const seasonIsAvailable = plexSeasons?.find((plexSeason) => plexSeason.index === season.seasonNumber);
        if (seasonIsAvailable) {
            seasonExistsInPlex = true;
        }
        return seasonExistsInPlex;
    }
    // Jellyfin
    async mediaExistsInJellyfin(media, is4k) {
        const ratingKey = media.jellyfinMediaId;
        const ratingKey4k = media.jellyfinMediaId4k;
        let existsInJellyfin = false;
        let preventSeasonSearch = false;
        // Check each jellyfin instance to see if the media still exists
        // If found, we will assume the media exists and prevent removal
        // We can use the cache we built when we fetched the series with mediaExistsInJellyfin
        try {
            let jellyfinMedia;
            if (ratingKey && !is4k) {
                jellyfinMedia = await this.jellyfinClient?.getItemData(ratingKey);
                if (media.mediaType === 'tv' && jellyfinMedia !== undefined) {
                    this.jellyfinSeasonsCache[ratingKey] =
                        await this.jellyfinClient?.getSeasons(ratingKey);
                }
            }
            if (ratingKey4k && is4k) {
                jellyfinMedia = await this.jellyfinClient?.getItemData(ratingKey4k);
                if (media.mediaType === 'tv' && jellyfinMedia !== undefined) {
                    this.jellyfinSeasonsCache[ratingKey4k] =
                        await this.jellyfinClient?.getSeasons(ratingKey4k);
                }
            }
            if (jellyfinMedia) {
                existsInJellyfin = true;
            }
        }
        catch (ex) {
            if (!ex.message.includes('404' || '500')) {
                existsInJellyfin = false;
                preventSeasonSearch = true;
                logger_1.default.debug(`Failure retrieving the ${is4k ? '4K' : 'non-4K'} ${media.mediaType === 'tv' ? 'show' : 'movie'} [TMDB ID ${media.tmdbId}] from Jellyfin.`, {
                    errorMessage: ex.message,
                    label: 'AvailabilitySync',
                });
            }
        }
        // Here we check each season in jellyfin for availability
        // If the API returns an error other than a 404,
        // we will have to prevent the season check from happening
        if (media.mediaType === 'tv') {
            const seasonsMap = new Map();
            if (!preventSeasonSearch) {
                const filteredSeasons = media.seasons.filter((season) => season[is4k ? 'status4k' : 'status'] === media_1.MediaStatus.AVAILABLE ||
                    season[is4k ? 'status4k' : 'status'] ===
                        media_1.MediaStatus.PARTIALLY_AVAILABLE);
                for (const season of filteredSeasons) {
                    const seasonExists = await this.seasonExistsInJellyfin(media, season, is4k);
                    if (seasonExists) {
                        seasonsMap.set(season.seasonNumber, true);
                    }
                }
            }
            return { existsInJellyfin, seasonsMap };
        }
        return { existsInJellyfin };
    }
    async seasonExistsInJellyfin(media, season, is4k) {
        const ratingKey = media.jellyfinMediaId;
        const ratingKey4k = media.jellyfinMediaId4k;
        let seasonExistsInJellyfin = false;
        // Check each jellyfin instance to see if the season exists
        let jellyfinSeasons;
        if (ratingKey && !is4k) {
            jellyfinSeasons = this.jellyfinSeasonsCache[ratingKey];
        }
        if (ratingKey4k && is4k) {
            jellyfinSeasons = this.jellyfinSeasonsCache[ratingKey4k];
        }
        const seasonIsAvailable = jellyfinSeasons?.find((jellyfinSeason) => jellyfinSeason.IndexNumber === season.seasonNumber);
        if (seasonIsAvailable) {
            seasonExistsInJellyfin = true;
        }
        return seasonExistsInJellyfin;
    }
}
const availabilitySync = new AvailabilitySync();
exports.default = availabilitySync;
