#!/bin/sh
# Copyright (c), 2024, Huawei Technologies Co, Ltd.
# Author: Zhihao Cheng <chengzhihao1@huawei.com>
#
# Test Description:
# For many kinds of flash, do following things
#  1. mount UBIFS
#  2. fsstress & powercut & unmount
#  3. fsck UBIFS
#  4. check UBIFS mounting result
# Running time: 1h

TESTBINDIR=/usr/lib/mtd-utils
source $TESTBINDIR/common.sh

function run_test()
{
	local simulator="$1";
	local size="$2";
	local peb_size="$3";
	local page_size="$4";
	local encryption=$5;

	echo "======================================================================"
	printf "%s" "$simulator: ${size}MiB PEB size ${peb_size}KiB"
	if [ "$simulator" = "nandsim" ]; then
		printf " %s" "page size ${page_size}Bytes"
	fi
	printf " $encryption\n"

	if [ "$simulator" = "nandsim" ]; then
		$TESTBINDIR/load_nandsim.sh "$size" "$peb_size" "$page_size" || echo "cannot load nandsim";
		mtdnum="$(find_mtd_device "$nandsim_patt")"
	elif [ "$simulator" = "mtdram" ]; then
		load_mtdram "$size" "$peb_size" || echo "cannot load mtdram"
		mtdnum="$(find_mtd_device "$mtdram_patt")"
	else
		fatal "$simulator is not supported"
	fi

	flash_eraseall /dev/mtd$mtdnum
	modprobe ubi mtd="$mtdnum,$page_size" || fatal "modprobe ubi fail"
	ubimkvol -N vol_test -m -n 0 /dev/ubi$UBI_NUM || fatal "mkvol fail"
	modprobe ubifs || fatal "modprobe ubifs fail"
	mount_ubifs $DEV $MNT || fatal "mount ubifs fail"
	if [[ "$encryption" == "encrypted" ]]; then
		encryption_gen_key
		encryption_set_key $MNT
	fi

	fsstress -d $MNT -l0 -p4 -n10000 &
	sleep $((RANDOM % 120))
	powercut

	ps -e | grep -w fsstress > /dev/null 2>&1
	while [ $? -eq 0 ]
	do
		killall -9 fsstress > /dev/null 2>&1
		sleep 1
		ps -e | grep -w fsstress > /dev/null 2>&1
	done

	while true
	do
		res=`mount | grep "$MNT"`
		if [[ "$res" == "" ]]
		then
			break;
		fi
		umount $MNT
		sleep 0.1
	done

	fsck.ubifs -a $DEV 2>&1 > $LOG_FILE
	res=$?
	cat $LOG_FILE
	if [[ $res != $FSCK_OK ]]
	then
		# Powercut during layout_leb_in_gaps may change index LEBs
		# without updating LPT.
		idx_log=`cat $LOG_FILE | grep "Inconsistent properties" | grep "is_idx 1"`
		# The lpt nodes could be parsed incorrectly because the lpt disk
		# layout is too simple. See details in
		# https://lore.kernel.org/linux-mtd/97ca7fe4-4ad4-edd1-e97a-1d540aeabe2d@huawei.com/
		lpt_log=`cat $LOG_FILE | grep "dbg_check_ltab_lnum: invalid empty space in LEB"`
		if [[ "$idx_log" == "" ]] && [[ "$lpt_log" == "" ]]; then
			fatal "fsck fail $res"
		fi
		if [[ $res != $FSCK_NONDESTRUCT ]]; then
			fatal "fsck fail $res"
		fi
	fi

	dmesg -c > /dev/null # powercut could reproduce error messages

	enable_chkfs

	mount_ubifs $DEV $MNT "noauthentication" "noatime"
	res=$?
	if [[ $res != 0 ]]
	then
		fatal "mount fail $res"
	fi

	if [[ "$encryption" == "encrypted" ]]; then
		encryption_set_key $MNT
	fi

	du -sh $MNT > /dev/null  # Make sure all files are accessible
	ret=$?
	if [[ $ret != 0 ]]; then
		fatal "Cannot access all files"
	fi
	check_err_msg

	umount $MNT
	res=$?
	if [[ $res != 0 ]]
	then
		fatal "unmount fail $res"
	fi

	check_err_msg

	modprobe -r ubifs
	modprobe -r ubi
	modprobe -r $simulator

	echo "----------------------------------------------------------------------"
}

check_fsstress
start_t=$(date +%s)
echo "Do powercut+fsck+mount test in kinds of flashes"
for simulator in "mtdram" "nandsim"; do
	for encryption in "encrypted" "noencrypted"; do
		run_test "$simulator" "16" "16" "512" $encryption
		run_test "$simulator" "64" "16" "512" $encryption
		run_test "$simulator" "128" "64" "2048" $encryption
		run_test "$simulator" "256" "128" "2048" $encryption
		run_test "$simulator" "512" "128" "2048" $encryption
		run_test "$simulator" "1024" "512" "2048" $encryption
	done
done
end_t=$(date +%s)
time_cost=$(( end_t - start_t ))
echo "Success, cost $time_cost seconds"
exit 0
